module net.BurtonRadons.dig.platform.fontSelector;

/** The font selector brings up an OS-specific dialog to select a font and its
  parameters.  The font returned from it can then be used throughout dig.

  Usage sample:

  @code
  Frame frame; // Our main frame.
  
  ...

  with (new FontSelector ())
  {
      Font f;

      effects = false; // Don't show underline, strikeout, and text color options.
      if ((f = run ()) !== null) // Display the selector.
      {
          // Succeeded, so set the font.
          frame.font (f);
          frame.paint ();
      }
  }
  @endcode

  */

class FontSelector
{
	private import net.BurtonRadons.dig.platform.base;
	private import net.BurtonRadons.dig.platform.control;
    private import net.BurtonRadons.dig.platform.font;
    private import net.BurtonRadons.dig.platform.windows;
    
    
    Font font; /**< The font to use as the default or null if there is none. */

    bit effects = true; /**< If true, show underline, strikeout, and text color options. */
    bit fixedPitchOnly = false; /**< If true, only show fixed pitch fonts. */

    float minSize; /**< Minimum font size, or nan for none. */
    float maxSize; /**< Maximum font size, or nan for none. */

    /** Empty constructor. */
    this ()
    {
    }

    /** Assigns the default font. */
    this (Font font)
    {
        this.font = font;
    }

    /** Run the selector and return the font selected or null if there was a
      * benign error.  If there was a serious error it will throw up.
      */

    Font run ()
    {
        TCHOOSEFONT info;
        LOGFONT log;

        info.lStructSize = info.size;
        info.hwndOwner = (_HANDLE) 0; /* Should set to any Frame */
        info.hDC = (_HANDLE) 0; /* Used with CF_PRINTERFONTS or CF_BOTH flags */
        info.lpLogFont = &log;
        if (this.font !== null)
            log = font.digPlatformLog;

        info.iPointSize = 0;
        info.Flags = 0;
        // CF_APPLY - provide an Apply button, that sends a message to the hook.
        // CF_ANSIONLY - obsolete.
        info.Flags |= CF_BOTH; // CF_BOTH - CF_SCREENFONTS | CF_PRINTERFONTS.
        // CF_TTONLY - provide TrueType fonts only.
        if (effects)
            info.Flags |= CF_EFFECTS;
        // CF_ENABLEHOOK - enable hook procedure.
        // CF_ENABLETEMPLATE - enable template.
        // CF_ENABLETEMPLATEHANDLE - enable preloaded template.
        if (fixedPitchOnly)
            info.Flags |= CF_FIXEDPITCHONLY;
        info.Flags |= CF_FORCEFONTEXIST; // Show an error message if the user selects a non-existent font.
        if (font !== null)
            info.Flags |= CF_INITTOLOGFONTSTRUCT;// - use the lpLogFont member for initial controls.
        if (!std.math.isnan (minSize) || !std.math.isnan (maxSize))
            info.Flags |= CF_LIMITSIZE;
        // CF_NOOEMFONTS - same as CF_NOVECTORFONTS.
        // CF_NOFACESEL - when using the LOGFONT, don't display an initial font name.
        info.Flags |= CF_NOSCRIPTSEL;// CF_NOSCRIPTSEL - disable the Script combo box.
        // CF_NOSTYLESEL - when using the LOGFONT, don't display an initial font style.
        // CF_NOSIZESEL - when using the LOGFONT, don't display an initial font size.
        // CF_NOSIMULATIONS - don't allow font simulations.
        // CF_NOVECTORFONTS - don't allow vector font selections.
        // CF_NOVERTFONTS - display only horizontally oriented fonts.
        // CF_PRINTERFONTS - list only fonts supported by the printer associated with the hDC member.
        // CF_SCALABLEONLY - display only scalable fonts (vector fonts, scalable printer fonts, TTF, etcetera).
        // CF_SCREENFONTS - display only screen fonts.
        // CF_SCRIPTSONLY - allow all scripts.
        // CF_SELECTSCRIPT - blah blah blah.
        // CF_SHOWHELP - display a Help button.  hwndOwner is sent the HE(_LPMSGSTRING messages when clicked.
        // CF_USESTYLE - use the lpszStyle member.
        // CF_WYSIWYG - only display fonts that are identical on printer and screen.
        info.rgbColors = 0; // Set to black initial color.
        info.lCustData = 0; // Sent to the hook procedure.
        info.lpfnHook = null; // The hook procedure, ignored unless if CF_ENABLEHOOK is set.
        info.lpTemplateName = null; // zstring template name if CF_ENABLETEMPLATE.
        info.hInstance = (_HANDLE) 0; // Used with CF_ENABLETEMPLATE and CF_ENABLETEMPLATEHANDLE.
        info.lpszStyle = null; // Style data used if CF_USESTYLE is non-null.
        info.nFontType = 0; // Font type returned, mask.
        if (!std.math.isnan (minSize) || !std.math.isnan (maxSize))
        {
            info.nSizeMin = std.math.isnan (minSize) ? 0 : minSize;
            info.nSizeMax = std.math.isnan (maxSize) ? 1024 * 1024 : maxSize;
        }

        if (!ChooseFontA (&info))
        {
            digPlatformCommDlgExtendedError ();
            return null;
        }

        return new Font (log);
    }
}